package Plugins::Spotty::Connect;

use strict;

use File::Path qw(mkpath);
use File::Slurp;
use File::Spec::Functions qw(catdir catfile);
use JSON::XS::VersionOneAndTwo;
use Scalar::Util qw(blessed);

use Slim::Utils::Log;
use Slim::Utils::Cache;
use Slim::Utils::Prefs;
use Slim::Utils::Timers;

use Plugins::Spotty::API qw(uri2url);

use constant CONNECT_HELPER_VERSION => '0.12.0';
use constant SEEK_THRESHOLD => 3;
use constant VOLUME_GRACE_PERIOD => 5;

my $cache = Slim::Utils::Cache->new();
my $prefs = preferences('plugin.spotty');
my $log = logger('plugin.spotty');

my $initialized;

sub init {
	my ($class) = @_;

	if (main::WEBUI && !$initialized) {
		require Plugins::Spotty::PlayerSettings;
		Plugins::Spotty::PlayerSettings->new();
	}

	return if $initialized;

	return unless $class->canSpotifyConnect('dontInit');

	require Plugins::Spotty::Connect::Context;

#                                                                |requires Client
#                                                                |  |is a Query
#                                                                |  |  |has Tags
#                                                                |  |  |  |Function to call
#                                                                C  Q  T  F
	Slim::Control::Request::addDispatch(['spottyconnect','_cmd'],
	                                                            [1, 0, 1, \&_connectEvent]
	);

	# listen to playlist change events so we know when Spotify Connect mode ends
	Slim::Control::Request::subscribe(\&_onNewSong, [['playlist'], ['newsong']]);

	# we want to tell the Spotify controller to pause playback when we pause locally
	Slim::Control::Request::subscribe(\&_onPause, [['playlist'], ['pause', 'stop']]);

	# we want to tell the Spotify about local volume changes
	Slim::Control::Request::subscribe(\&_onVolume, [['mixer'], ['volume']]);

	require Plugins::Spotty::Connect::DaemonManager;
	Plugins::Spotty::Connect::DaemonManager->init();

	$initialized = 1;
}

sub canSpotifyConnect {
	my ($class, $dontInit) = @_;

	# we need a minimum helper application version
	if ( !Slim::Utils::Versions->checkVersion(Plugins::Spotty::Plugin->getHelperVersion(), CONNECT_HELPER_VERSION, 10) ) {
		$log->error("Cannot support Spotty Connect, need at least helper version " . CONNECT_HELPER_VERSION);
		return;
	}

	__PACKAGE__->init() unless $initialized || $dontInit;

	return 1;
}

sub isSpotifyConnect {
	my ( $class, $client ) = @_;

	return unless $client;
	$client = $client->master;
	my $song = $client->playingSong();

	return unless $client->pluginData('SpotifyConnect');

	return $client->pluginData('newTrack') || _contextTime($song) ? 1 : 0;
}

sub _contextTime {
	my ($song) = @_;

	return unless $song && $song->pluginData('context');
	return $song->pluginData('context')->time() || 0;
}

sub setSpotifyConnect {
	my ( $class, $client, $context ) = @_;

	return unless $client;

	$client = $client->master;
	my $song = $client->playingSong();

	# state on song: need to know whether we're currently in Connect mode. Is lost when new track plays.
	$song->pluginData('context') || $song->pluginData( context => Plugins::Spotty::Connect::Context->new($class->getAPIHandler($client)) );
	$song->pluginData('context')->update($context);
	$song->pluginData('context')->time(time());

	# state on client: need to know whether we've been in Connect mode. If this is set, then we've been playing from Connect, but are no more.
	$client->pluginData( SpotifyConnect => 1 );
}

sub getNextTrack {
	my ($class, $song, $successCb, $errorCb) = @_;

	my $client = $song->master;

	my $spotty = $class->getAPIHandler($client);

	if ( $client->pluginData('newTrack') ) {
		main::INFOLOG && $log->is_info && $log->info("Don't get next track as we got called by a play track event from spotty");

		# XXX - how to deal with context here?
		$spotty->player(sub {
				$class->setSpotifyConnect($client, $_[0]);
				$client->pluginData( newTrack => 0 );
				$successCb->();
		});
	}
	else {
		main::INFOLOG && $log->is_info && $log->info("We're approaching the end of a track - get the next track");

		$client->pluginData( newTrack => 1 );

		# add current track to the history
		$song->pluginData('context')->addPlay($song->track->url);

		# for playlists and albums we can know the last track. In this case no further check would be required.
		if ( $song->pluginData('context')->isLastTrack($song->track->url) ) {
			$class->_delayedStop($client);
			$successCb->();
			return;
		}

		$spotty->playerNext(sub {
			$spotty->player(sub {
				my ($result) = @_;

				if ( $result && ref $result && (my $uri = $result->{item}->{uri}) ) {
					main::INFOLOG && $log->is_info && $log->info("Got a new track to be played next: $uri");

					$uri = uri2url($uri);

					# stop playback if we've played this track before. It's likely trying to start over.
					if ( $song->pluginData('context')->hasPlay($uri) && !($result->{repeat_state} && $result->{repeat_state} eq 'on')) {
						$class->_delayedStop($client);
					}
					else {
						$song->track->url($uri);
						# $song->streamUrl($uri); # wouldn't continue to next track
						$class->setSpotifyConnect($client, $result);
					}

					$successCb->();
				}
			});
		});
	}
}

sub _delayedStop {
	my ($class, $client) = @_;

	# set a timer to stop playback at the end of the track
	my $remaining = $client->controller()->playingSongDuration() - Slim::Player::Source::songTime($client);
	main::INFOLOG && $log->is_info && $log->info("Stopping playback in ${remaining}s, as we have likely reached the end of our context (playlist, album, ...)");

	Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $remaining, sub {
		$client->pluginData( newTrack => 0 );
		$class->getAPIHandler($client)->playerPause(sub {
			$client->execute(['stop']);
		}, $client->id);
	});
}

sub _onNewSong {
	my $request = shift;

	return if $request->source && $request->source eq __PACKAGE__;

	my $client  = $request->client();
	return if !defined $client;
	$client = $client->master;

	if (__PACKAGE__->isSpotifyConnect($client)) {
		# if we're in Connect mode and have seek information, go there
		if ( $client && (my $progress = $client->pluginData('progress')) ) {
			$client->pluginData( progress => 0 );
			$client->execute( ['time', int($progress)] );
		}

		return;
	}

	return unless $client->pluginData('SpotifyConnect');

	main::INFOLOG && $log->is_info && $log->info("Got a new track event, but this is no longer Spotify Connect");
	$client->playingSong()->pluginData( context => 0 );
	$client->pluginData( SpotifyConnect => 0 );
	__PACKAGE__->getAPIHandler($client)->playerPause(undef, $client->id);
}

sub _onPause {
	my $request = shift;

	return if $request->source && $request->source eq __PACKAGE__;

	# no need to pause if we unpause
	return if $request->isCommand([['playlist'],['pause']]) && !$request->getParam('_newvalue');

	my $client  = $request->client();
	return if !defined $client;
	$client = $client->master;

	# ignore pause while we're fetching a new track
	return if $client->pluginData('newTrack');

	return if !__PACKAGE__->isSpotifyConnect($client);

	if ( $request->isCommand([['playlist'],['stop','pause']]) && _contextTime($client->playingSong()) > time() - 5 ) {
		main::INFOLOG && $log->is_info && $log->info("Got a stop event within 5s after start of a new track - do NOT tell Spotify Connect controller to pause");
		return;
	}

	main::INFOLOG && $log->is_info && $log->info("Got a pause event - tell Spotify Connect controller to pause, too");
	__PACKAGE__->getAPIHandler($client)->playerPause(undef, $client->id);
}

sub _onVolume {
	my $request = shift;

	return if $request->source && $request->source eq __PACKAGE__;

	my $client  = $request->client();
	return if !defined $client;
	$client = $client->master;

	return if !__PACKAGE__->isSpotifyConnect($client);

	my $volume = $client->volume;

	# buffer volume change events, as they often come in bursts
	Slim::Utils::Timers::killTimers($client, \&_bufferedSetVolume);
	Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + 0.5, \&_bufferedSetVolume, $volume);
}

sub _bufferedSetVolume {
	my ($client, $volume) = @_;
	main::INFOLOG && $log->is_info && $log->info("Got a volume event - tell Spotify Connect controller to adjust volume, too: $volume");
	__PACKAGE__->getAPIHandler($client)->playerVolume(undef, $client->id, $volume);
}

sub _connectEvent {
	my $request = shift;
	my $client = $request->client()->master;

	if ( $client->pluginData('newTrack') ) {
		$client->pluginData( newTrack => 0 );
		return;
	}

	my $cmd = $request->getParam('_cmd');

	main::INFOLOG && $log->is_info && $log->info("Got called from spotty helper: $cmd");

	if ( $cmd eq 'volume' && !($request->source && $request->source eq __PACKAGE__) ) {
		my $volume = $request->getParam('_p2') || return;

		# sometimes volume would be reset to a default 50 right after the daemon start - ignore
		if ( $volume == 50 && Plugins::Spotty::Connect::DaemonManager->uptime($client->id) < VOLUME_GRACE_PERIOD ) {
			main::INFOLOG && $log->is_info && $log->info("Ignoring initial volume reset right after daemon start");
			return;
		}

		main::INFOLOG && $log->is_info && $log->info("Set volume to $volume");

		# we don't let spotty handle volume directly to prevent getting caught in a call loop
		my $request = Slim::Control::Request->new( $client->id, [ 'mixer', 'volume', $volume ] );
		$request->source(__PACKAGE__);
		$request->execute();

		return;
	}

	my $spotty = __PACKAGE__->getAPIHandler($client);

	$spotty->player(sub {
		my ($result) = @_;

		my $song = $client->playingSong();
		my $streamUrl = ($song ? $song->streamUrl : '') || '';
		$streamUrl =~ s/\/\///;

		$song && $song->pluginData('context') && $song->pluginData('context')->update($result);

		$result ||= {};

		main::DEBUGLOG && $log->is_debug && $log->debug("Current Connect state: \n" . Data::Dump::dump($result, $cmd));

		# in case of a change event we need to figure out what actually changed...
		if ( $cmd =~ /change/ && $result && ref $result && (($streamUrl ne $result->{track}->{uri} && $result->{is_playing}) || !__PACKAGE__->isSpotifyConnect($client)) ) {
			main::INFOLOG && $log->is_info && $log->info("Got a $cmd event, but actually this is a play next track event");
			$cmd = 'start';
		}

		if ( $cmd eq 'start' && $result->{track} ) {
			if ( $streamUrl ne $result->{track}->{uri} || !__PACKAGE__->isSpotifyConnect($client) ) {
				main::INFOLOG && $log->is_info && $log->info("Got a new track to be played: " . $result->{track}->{uri});

				# Sometimes we want to know whether we're in Spotify Connect mode or not
				$client->pluginData( SpotifyConnect => 1 );
				$client->pluginData( newTrack => 1 );

				my $request = $client->execute( [ 'playlist', 'play', $result->{track}->{uri} ] );
				$request->source(__PACKAGE__);

				# sync volume up to spotify if we just got connected
				if ( !$client->pluginData('SpotifyConnect') ) {
					$spotty->playerVolume(undef, $client->id, $client->volume);
				}

				# on interactive Spotify Connect use we're going to reset the play history.
				# this isn't really solving the problem of lack of context. But it's better than nothing...
				$song && $song->pluginData('context') && $song->pluginData('context')->reset();

				$result->{progress} ||= ($result->{progress_ms} / 1000) if $result->{progress_ms};

				# if status is already more than 10s in, then do seek
				if ( $result->{progress} && $result->{progress} > 10 ) {
					$song && $client->pluginData( progress => $result->{progress} );
				}
			}
			elsif ( !$client->isPlaying ) {
				main::INFOLOG && $log->is_info && $log->info("Got to resume playback");
				__PACKAGE__->setSpotifyConnect($client, $result);
				my $request = Slim::Control::Request->new( $client->id, ['play'] );
				$request->source(__PACKAGE__);
				$request->execute();
			}
		}
		elsif ( $cmd eq 'stop' && $result->{device} ) {
			my $clientId = $client->id;

			# if we're playing, got a stop event, and current Connect device is us, then pause
			if ( $client->isPlaying && ($result->{device}->{id} eq Plugins::Spotty::API->idFromMac($clientId) || $result->{device}->{name} eq $client->name) && __PACKAGE__->isSpotifyConnect($client) ) {
				main::INFOLOG && $log->is_info && $log->info("Spotify told us to pause: " . $client->id);

				my $request = Slim::Control::Request->new( $client->id, ['pause', 1] );
				$request->source(__PACKAGE__);
				$request->execute();
			}
			elsif ( $client->isPlaying && ($result->{device}->{id} ne Plugins::Spotty::API->idFromMac($clientId) && $result->{device}->{name} ne $client->name) && __PACKAGE__->isSpotifyConnect($client) ) {
				main::INFOLOG && $log->is_info && $log->info("Spotify told us to pause, but current player is no longer the Connect target");

				my $request = Slim::Control::Request->new( $client->id, ['pause', 1] );
				$request->source(__PACKAGE__);
				$request->execute();

				# reset Connect status on this device
				$client->playingSong()->pluginData( context => 0 );
				$client->pluginData( SpotifyConnect => 0 );
			}
			# if we're playing, got a stop event, and current Connect device is NOT us, then
			# disable Connect and let the track end
			elsif ( $client->isPlaying ) {
				main::INFOLOG && $log->is_info && $log->info("Spotify told us to pause, but current player is not Connect target");
				$client->playingSong()->pluginData( context => 0 );
				$client->pluginData( SpotifyConnect => 0 );
			}
		}
		elsif ( $cmd eq 'change' ) {
			# seeking event from Spotify - we would only seek if the difference was larger than x seconds, as we'll never be perfectly in sync
			if ( $client->isPlaying && abs($result->{progress} - Slim::Player::Source::songTime($client)) > SEEK_THRESHOLD ) {
				$client->execute( ['time', int($result->{progress})] );
			}
		}
		elsif (main::INFOLOG && $log->is_info) {
			$log->info("Unknown command called? $cmd\n" . Data::Dump::dump($result));
		}
	});
}

=pod
	Here we're overriding some of the default handlers. In Connect mode, when discovery is enabled,
	we could be streaming from any account, not only those configured in Spotty. Therefore we need
	to use different cache folders with credentials. Use the currently set in Spotty as default,
	but read actual value whenever accessing the API. We won't keep these credentials around, to
	prevent using a visitor's account.
=cut
sub getAPIHandler {
	my ($class, $client) = @_;

	return unless $client;

	my $api;

	my $cacheFolder = $class->cacheFolder($client);
	my $credentialsFile = catfile($cacheFolder, 'credentials.json');

	my $credentials = eval {
		from_json(read_file($credentialsFile));
	};

	if ( !$@ && $credentials || ref $credentials && $credentials->{auth_data} ) {
		$api = Plugins::Spotty::API->new({
			client => $client,
			cache => $cacheFolder,
			username => $credentials->{username},
		});
	}

	return $api || Plugins::Spotty::Plugin->getAPIHelper($client);
}

sub cacheFolder {
	my ($class, $clientId) = @_;

	$clientId = $clientId->id if $clientId && blessed $clientId;

	my $cacheFolder = Plugins::Spotty::Plugin->cacheFolder( Plugins::Spotty::Plugin->getAccount($clientId) );

	# create a temporary account folder with the player's MAC address
	if ( Plugins::Spotty::Plugin->canDiscovery() && !$prefs->get('disableDiscovery') ) {
		my $id = $clientId;
		$id =~ s/://g;

		my $playerCacheFolder = catdir(preferences('server')->get('cachedir'), 'spotty', $id);
		mkpath $playerCacheFolder unless -e $playerCacheFolder;

		if ( !-e catfile($playerCacheFolder, 'credentials.json') ) {
			require File::Copy;
			File::Copy::copy(catfile($cacheFolder, 'credentials.json'), catfile($playerCacheFolder, 'credentials.json'));
		}
		$cacheFolder = $playerCacheFolder;
	}

	return $cacheFolder
}

sub shutdown {
	if ($initialized) {
		Plugins::Spotty::Connect::DaemonManager->shutdown();
	}
}

1;