package Plugins::PlayHistory::Plugin;

use strict;
use base qw(Slim::Plugin::OPMLBased);

use Digest::MD5 qw(md5_hex);

use Slim::Utils::DateTime;
use Slim::Utils::Log;
use Slim::Utils::Prefs;
use Slim::Utils::Strings qw(cstring);
use Slim::Utils::Timers;

use Plugins::PlayHistory::Menu;

use constant DELAY_BEFORE_LOGGING => 10;

my $prefs = preferences('plugin.playhistory');
my $serverprefs = preferences('server');

my $log = Slim::Utils::Log->addLogCategory( {
	category     => 'plugin.playhistory',
	defaultLevel => 'ERROR',
	description  => 'PLUGIN_PLAYHISTORY',
} );

my ($buffer, $ignoreList);


sub initPlugin {
	my $class = shift;
	
	$buffer = Plugins::PlayHistory::Buffer->new();

	$prefs->init({
		buffer => [],
		bufferSize => 200,
		sortDescending => 1,
		ignoreKeywords => 'srf3, radio paradise, swr',
	});
	
	$prefs->setChange( sub {
		$buffer->validate()
	}, 'bufferSize');
	
	$prefs->setChange( sub {
		$ignoreList = '';
	}, 'ignoreKeywords');
	
	$prefs->migrate(1, sub {
		my $buffer = $prefs->get('buffer');
		
		$buffer = [ map {
			$_->{clients} = [ $_->{clients} ] if $_->{clients} && ref $_->{clients} eq 'SCALAR';
			$_->{clients} = [] unless $_->{clients} && ref $_->{clients} eq 'ARRAY';
			$_;
		} @$buffer ];
		
		$prefs->set('buffer', $buffer);
	});

	if (main::WEBUI) {
		require Plugins::PlayHistory::Settings;
		Plugins::PlayHistory::Settings->new();
	}
	
	Plugins::PlayHistory::Menu->init();
	
	# Subscribe to new song events
	Slim::Control::Request::subscribe( \&newsongCallback, [['playlist', 'newmetadata']] );

	Slim::Menu::TrackInfo->registerInfoProvider( playhistory => (
		after => 'top',
		func  => \&nowPlayingInfoMenu,
	) );

	$class->SUPER::initPlugin(
		feed   => \&handleFeed,
		tag    => 'playhistory',
		is_app => 1,
	);
}

sub getDisplayName { 'PLUGIN_PLAYHISTORY' }

sub handleFeed {
	my ($client, $cb, $params, $args) = @_;
	
	my $items = [];
	
	if ( !$args || !$args->{client} ) {
		push @$items, {
			type => 'link',
			name => cstring($client, 'ALL'),
			url  => \&handleFeed,
			passthrough => [{
				client => '_ALL_'
			}],
		};
		
		foreach my $clientId ( sort keys %{ $buffer->getPlayers || {} } ) {
			my $c = Slim::Player::Client::getClient($clientId);
			my $playername = $clientId;
			
			if ($c) {
				$playername = $c->name;
			}
			else {
				eval {
					$playername = Slim::Utils::Prefs::Client->new($serverprefs, $clientId)->get('playername');
				};
				
				$@ && $log->warn("Failed to get player name from prefs: $@");
			}
			
			push @$items, {
				type => 'link',
				name => $playername || $clientId,
				url  => \&handleFeed,
				passthrough => [{
					client => $clientId
				}]
			};
		}
	}

	if ( scalar @$items < 3 ) {
		$items = [];
		
		foreach ( @{ $buffer->getSorted($args->{client}) } ) {
			my $timestamp = Slim::Utils::DateTime::shortDateF($_->{ts}) . ' ' . Slim::Utils::DateTime::timeF($_->{ts});
			
			my $url = $_->{url};
			
			my $item = {
				type  => 'link',
				name  => $timestamp . ' - ' .  $_->{title} . ' - ' . $_->{artist},
				line1 => $timestamp . ' - ' .  $_->{title},
				line2 => $_->{artist},
				url   => sub {
					my ($client, $cb, $params, $args) = @_;
					$cb->( Plugins::PlayHistory::Menu->menu( 
						$client, 
						$args->{id}, 
						{ menuMode => 'playhistory' } 
					) );
				},
				passthrough => [{
					id => $_->{_id}
				}],
			};
			
			if ($_->{remote_title}) {
				$item->{name}  .= ' (' . $_->{remote_title} . ')';
				$item->{line2} .= ' - ' . $_->{remote_title};
			}
			
			if ($_->{artwork_url}) {
				$item->{image} = $_->{artwork_url};
			}
			elsif ($_->{coverid}) {
				$item->{image} = 'music/' . $_->{coverid} . '/cover';
				$item->{icon}  = $_->{coverid};
			}
			
			$item->{name}  =~ s/^ - //;
			$item->{line1} =~ s/^ - //;
			$item->{line2} =~ s/^ - //;
	
			push @$items, $item;
		}
	}
	
	$cb->({
		items => $items
	});
}

sub nowPlayingInfoMenu {
	my ( $client, $url, $track, $remoteMeta, $tags ) = @_;

	return unless $client && $tags;

	return {
		name => $client->string(getDisplayName()),
		type => 'link',
		url  => \&handleFeed,
		isContextMenu => 1,
	};
}

sub getPlayedItem {
	my ($class, $id) = @_;
	return $buffer->getItem($id);
}

sub newsongCallback {
	my $request = shift;
	
	return unless $request->isCommand([['playlist'], ['newsong']]) || $request->isCommand([['newmetadata']]);

	my $client  = $request->client() || return;
	
	if (!$client->isPlaying()) {
		main::DEBUGLOG && $log->debug('Client is inactive. Ignore.');
		return;
	}
	
	Slim::Utils::Timers::killTimers($client, \&_logTrack);
	Slim::Utils::Timers::setTimer($client, time + DELAY_BEFORE_LOGGING, \&_logTrack);
}

sub _logTrack {
	my $client = shift;
	
	my $url = Slim::Player::Playlist::url($client);
	
	return unless $url;

	my $request = Slim::Control::Request::executeRequest($client, ['songinfo', 0, 99, 'url:' . $url]);
	
	if ($request) {
		#main::DEBUGLOG && $log->is_debug && $log->debug(Data::Dump::dump($request->getResults()));
		
		my $songinfo;
		
		eval {
			$songinfo = $request->getResults()->{songinfo_loop};
		};
		
		if ($@) {
			$log->warn("Failed to get song information: $@\n" . Data::Dump::dump($request->getResults()));
			return;
		}

		my $meta = {};
		foreach (@$songinfo) {
			my ($k, $v) = each %$_;
			
			if ($k =~ /^(?:id|artist|title|remote_title|artwork_url|coverid)$/) {
				$meta->{$k} = $v || '';
			}
		}
		
		$ignoreList ||= join('|', map {
			s/^\s*//;
			s/\s*$//;
			$_;
		} split(/,/, $prefs->get('ignoreKeywords') || ''));
		
		# skip tracks based on blacklisted keywords
		if ( defined $ignoreList && $ignoreList ne '' && ($meta->{title} =~ /\b(?:$ignoreList)\b/i || $meta->{artist} =~ /\b(?:$ignoreList)\b/i
				# filtering by remote_title is optional, as this often is the station name, not track information
				|| ($prefs->get('filterRemoteTitle') && $meta->{remote_title} && $meta->{remote_title} =~ /\b(?:$ignoreList)\b/i)) ) {
			main::DEBUGLOG && $log->is_debug && $log->debug("Ignoring track, its title or artist matches our skiplist: '$ignoreList' - " . $meta->{title} . ' - ' . $meta->{artist});
			return;
		}
		
		$meta->{url}    = $url;
		$meta->{ts}     = time;
		$meta->{_id}    = md5_hex($meta->{ts} . $url);
		$meta->{client} = $client->id;
		$meta->{clients} = [ $meta->{client}, map { $_->id } Slim::Player::Sync::slaves($client) ];

		$buffer->addItem($meta);
	}
}

1;


# wrapper package around the buffer pref - make sure sort order, structure etc. are ok
package Plugins::PlayHistory::Buffer;

sub new {
	my $class = shift;
	
	my $self = {
		_buffer => $prefs->get('buffer') || []
	};
	
	validate($self);
	
	return bless $self, $class;
}

sub get {
	my $class = shift;
	return $class->{_buffer};
}

sub getSorted {
	my ($class, $clientId) = @_;
	
	my $buffer = $class->{_buffer};
	
	if ( $clientId && $clientId ne '_ALL_' ) {
		$buffer = [ grep {
			grep /$clientId/i, @{ $_->{clients} || [] }
		} @$buffer ];
	}
	
	if (!$prefs->get('sortDescending')) {
		return [ sort { $a->{ts} <=> $b->{ts} } @{$buffer} ];
	}

	return $buffer || [];
}

sub addItem {
	my ($class, $item) = @_;
	
	my $clientId = delete $item->{client};

	# skip same track?
	my $previous = $class->getSorted($clientId)->[0];
	if ( $previous && $previous->{url} eq $item->{url} 
		&& $previous->{artist} eq $item->{artist} 
		&& $previous->{title} eq $item->{title}
	) {

		if ( !grep /$clientId/i, @{ $previous->{clients} } ) {
			main::DEBUGLOG && $log->is_debug && $log->debug("Updating track - adding client '$clientId': " . $item->{title} . ' - ' . $item->{artist});
			push @{ $previous->{clients} }, $clientId;
		}
		else {
			main::DEBUGLOG && $log->is_debug && $log->debug("Ignoring track - same as previous: " . $item->{title} . ' - ' . $item->{artist});
			return;
		}
		
	}
	else {
		$item->{clients} ||= [ $clientId ];
		main::DEBUGLOG && $log->is_debug && $log->debug("Adding item: \n" . Data::Dump::dump($item));
		unshift @{$class->{_buffer}}, $item;
	}
	
	$class->validate();
}

sub getItem {
	my ($class, $id) = @_;
	
	return unless $id;
	
	my ($item) = grep {
		$_->{_id} eq $id;
	} @{$class->{_buffer}};
	
	return $item;
}

sub getItemAt {
	my ($class, $index) = @_;
	
	$class->{_buffer}->[$index || 0];
}

sub getPlayers {
	my ($class) = @_;
	
	my $players;
	
	foreach my $item ( @{$class->{_buffer}} ) {
		foreach my $player ( @{$item->{clients}} ) {
			$players->{$player}++;
		}
	}
	
	return $players;
}

sub validate {
	my $class = shift;
	
	# initialize buffer from prefs if we haven't yet
	$class->{_buffer} ||= $prefs->get('buffer');
	
	# don't log dupes...
	my %seen;
	$class->{_buffer} = [ grep {
		!$seen{$_->{_id}}++
	} @{$class->{_buffer}} ];

	# sort descending
	$class->{_buffer} = [ sort { $b->{ts} <=> $a->{ts} } @{$class->{_buffer}} ];

	# truncate at max length
	my $bufferSize = $prefs->get('bufferSize') || 200;
	if ( @{$class->{_buffer}} > $bufferSize ) {
		$class->{_buffer} = [ splice(@{$class->{_buffer}}, 0, $bufferSize) ];
	}
	
	$prefs->set('buffer', $class->{_buffer});
}

1;
